################################################################
# Replication of:                                              #
#                                                              #
# Yordanova, Nikoleta; Mariyana Angelova; Roni Lehrer;         #
# Moritz Osnabrgge and Sander Renes (2020)                    # 
# "Swaying citizen support for EU membership:                  #
# Evidence from a survey experiment of German voters"          #
# European Union Politics                                      #
#                                                              #
# Created by Roni Lehrer, 18 March 2020                        #
#                                                              #
# see end of script for information on R and packages versions #
################################################################

rm(list=ls())
library(dplyr)
library(reshape2)
library(ggplot2)
library(readstata13)
library(MASS)
library(stargazer)
library(data.table)
library(car)
library(MASS)

#function to get difference between two OLS coefficients and their CIs
difference <- function(reg, smaller, greater) {
 betas <- mvrnorm(1000, coef(reg), vcov(reg))
 diff <- betas[,greater]-betas[,smaller]
 return(list(mean=mean(diff), lower=quantile(diff, .025), upper=quantile(diff, .975)))
}

##############################################################################################################
# If you are using "data_for_replication.Rdata" to replicate the study, please use the following code.       #
# You can replicate "data_for_replication.Rdata" from original GIP data by running the code in lines 39-202. #
##############################################################################################################
load("data_for_replication.Rdata")



## #################################################################################################
## # The datasets for this replication are Waves 24, 25, and 28 of the German Internet Panel.      #
## # They are stored at the Gesis Data Archive as:                                                 #
## #                                                                                               #
## # ****************                                                                              #
## # *** WAVE 24 ****                                                                              #
## # ****************                                                                              #
## # Blom, Annelies G., Bruch, Christian, Felderer, Barbara, Gebhard, Franziska, Herzing, Jessica, #
## # Krieger, Ulrich, & SFB 884 Political Economy of Reforms, Universitt Mannheim (2017).       #
## # German Internet Panel, Welle 24 (Juli 2016). GESIS Datenarchiv, Kln.                         #
## # ZA6756 Datenfile Version 2.0.0 (2017).                                                        #
## #                                                                                               #
## # Internet: http://dx.doi.org/1.12844                                                           #
## # Gesis Study Number: ZA6756                                                                    #
## #                                                                                               #
## # ****************                                                                              #
## # *** WAVE 25 ****                                                                              #
## # ****************                                                                              #
## # Blom, Annelies G., Bruch, Christian, Felderer, Barbara, Gebhard, Franziska, Herzing, Jessica, #
## # Krieger, Ulrich, & SFB 884 Political Economy of Reforms, Universitt Mannheim (2017).       #
## # German Internet Panel, Welle 25 - Core Study (September 2016). GESIS Datenarchiv, Kln.       #
## # ZA6757 Datenfile Version 1.0.0 (2017).                                                        #
## #                                                                                               #
## # Internet: http://dx.doi.org/1.12755                                                           #
## # Gesis Study Number: ZA6757                                                                    #
## #                                                                                               #
## # ****************                                                                              #
## # *** WAVE 28 ****                                                                              #
## # ****************                                                                              #
## # Blom, Annelies G., Bruch, Christian, Felderer, Barbara, Gebhard, Franziska, Herzing, Jessica, #
## # Krieger, Ulrich, & SFB 884 Political Economy of Reforms, Universitt Mannheim (2017).       #
## # German Internet Panel, Welle 28 (Mrz 2017). GESIS Datenarchiv, Kln.                         #
## # ZA6902 Datenfile Version 1.0.0 (2017).                                                        #
## #                                                                                               #
## # Internet: http://dx.doi.org/1.12890                                                           #
## # Gesis Study Number: ZA6902                                                                    #
## #################################################################################################



## ##Load data and keep relevant variables (if needed, set working directory first)
## GIP_W24_V1 <- read.dta13("ZA6756_v2-0-0.dta")
## GIP_W28_V1 <- read.dta13("ZA6902_v1-0-0.dta")
## GIP_W25_V1 <- read.dta13("ZA6757_v1-0-0.dta")

## GIP_W28_experiment <- dplyr::select(GIP_W28_V1, id_g, ZJ28020, expZJ28020)
## GIP_W24_experiment <- dplyr::select(GIP_W24_V1, id_g, ZJ24020,  CG24040, CG24041, gender_15, year_of_birth_cat,
##                                     educ_school_15, state)
## total_experiment<- merge(GIP_W28_experiment, GIP_W24_experiment, by="id_g", all.x=TRUE)
## GIP_W25_experiment <- dplyr::select(GIP_W25_V1,id_g, AA25039,AA25040a)
## total_experiment<- merge(total_experiment, GIP_W25_experiment, by="id_g", all.x=TRUE)
## rm(GIP_W24_V1)
## rm(GIP_W28_V1)
## rm(GIP_W25_V1)

## #####################
## # Data manipulation #
## #####################
## total_experiment$ZJ24020 <- ifelse(total_experiment$ZJ24020%in%c(1,2,-99)==F,
##                                    NA, total_experiment$ZJ24020)
## total_experiment$ZJ28020 <- ifelse(total_experiment$ZJ28020%in%c(1,2,-99)==F,
##                                    NA, total_experiment$ZJ28020)

## #keep only relevant cases
## total_experiment <- total_experiment[!is.na(total_experiment$ZJ28020),]
## total_experiment <- total_experiment[!is.na(total_experiment$ZJ24020),]

## #Recode control group to 0, treatments stay 1 to 8.
## total_experiment$expZJ28020=ifelse(total_experiment$expZJ28020<9, total_experiment$expZJ28020, 0)

## #Create treatment factor variable
## total_experiment$trt.factor <- factor(total_experiment$expZJ28020, labels=c("Control",
## "Political Pro", "Political Con",
## "Cultural Pro", "Cultural Con",
## "Economic Pro", "Economic Con",
## "Peace Pro", "Peace Con"))


## #Three point scale DV (needed for summary statistics)
## total_experiment$new.scale24 <- ifelse(total_experiment$ZJ24020==1, 1,
##                                        ifelse(total_experiment$ZJ24020==2, -1,
##                                               ifelse(total_experiment$ZJ24020==-99, 0, NA)))
## total_experiment$new.scale28 <- ifelse(total_experiment$ZJ28020==1, 1,
##                                        ifelse(total_experiment$ZJ28020==2, -1,
##                                               ifelse(total_experiment$ZJ28020==-99, 0, NA)))

## #Two point scale DVs (needed for analyses)
## total_experiment$remain24 <- ifelse(total_experiment$ZJ24020==1, 1,
##                                       ifelse(total_experiment$ZJ24020==2, 0,
##                                              ifelse(total_experiment$ZJ24020==-99, 0, NA)))
## total_experiment$leave24 <- ifelse(total_experiment$ZJ24020==2, 1,
##                                       ifelse(total_experiment$ZJ24020==1, 0,
##                                              ifelse(total_experiment$ZJ24020==-99, 0, NA)))
## total_experiment$DK24<- ifelse(total_experiment$ZJ24020==-99, 1,
##                                       ifelse(total_experiment$ZJ24020==1, 0,
##                                              ifelse(total_experiment$ZJ24020==2, 0, NA)))
## total_experiment$remain28 <- ifelse(total_experiment$ZJ28020==1, 1,
##                                       ifelse(total_experiment$ZJ28020==2, 0,
##                                              ifelse(total_experiment$ZJ28020==-99, 0, NA)))
## total_experiment$leave28 <- ifelse(total_experiment$ZJ28020==2, 1,
##                                       ifelse(total_experiment$ZJ28020==1, 0,
##                                       ifelse(total_experiment$ZJ28020==-99, 0, NA)))
## total_experiment$DK28<- ifelse(total_experiment$ZJ28020==-99, 1,
##                                       ifelse(total_experiment$ZJ28020==1, 0,
##                                              ifelse(total_experiment$ZJ28020==2, 0, NA)))

## #Delta dependent variables used in regressions
## total_experiment$Dif_remain =total_experiment$remain28 - total_experiment$remain24
## total_experiment$Dif_leave = total_experiment$leave28 - total_experiment$leave24
## total_experiment$Dif_dk = total_experiment$DK28 - total_experiment$DK24

## #pro EU sentiment
## total_experiment$pro <- ifelse(total_experiment$CG24040<0, NA, total_experiment$CG24040)
## total_experiment$pro.dum<- ifelse(is.na(total_experiment$CG24040), NA,
##                                        ifelse(total_experiment$CG24040>=4, 1,
##                                                ifelse(total_experiment$CG24040<=2, 0, NA)))

## #Schooling (up to 10 years)
## total_experiment$schoolupto10years<- ifelse(total_experiment$educ_school_15<0, NA,
##                                             total_experiment$educ_school_15)
## total_experiment$schoolupto10years<- ifelse(total_experiment$educ_school_15<5, 1, 0)

## #Age categories
## total_experiment$age <- ifelse(total_experiment$year_of_birth_cat<0 , NA,
##                                total_experiment$year_of_birth_cat)
## total_experiment$age <- ifelse(total_experiment$year_of_birth_cat %in% c(1,2,3) , 4,
##                                total_experiment$age) #67+
## total_experiment$age <- ifelse(total_experiment$year_of_birth_cat %in% c(4,5,6,7) , 3,
##                                total_experiment$age) #47-66
## total_experiment$age <- ifelse(total_experiment$year_of_birth_cat %in% c(8,9) , 2,
##                                total_experiment$age) #37-46
## total_experiment$age <- ifelse(total_experiment$year_of_birth_cat %in% c(10,11) , 1,
##                                total_experiment$age) #27-36
## total_experiment$age <- ifelse(total_experiment$year_of_birth_cat %in% c(12,13) , 0,
##                                total_experiment$age) # -27

## #East Germany dummy
## total_experiment$east <- ifelse(total_experiment$state<11 , 0, 1) #east Germany

## #anti-EU party
## total_experiment$anti_party <- ifelse(total_experiment$AA25039<0|total_experiment$AA25039==10, NA, total_experiment$AA25039)
## total_experiment$anti_party <- ifelse(total_experiment$anti_party%in%c(7,11,8), 1, #AfD, Linke, NPD
##                               ifelse(total_experiment$anti_party%in%c(1,2,10,-98,-99), NA, #missings
##                                      0)) #pro-EU parties

## #LR self.placement
## total_experiment$LR <- ifelse(total_experiment$AA25040a<0, NA, total_experiment$AA25040a)

## #Female dummy
## total_experiment$female=ifelse(total_experiment$gender_15<1, NA, total_experiment$gender_15-1)

## #EU extremists: people feeling very positively or very negatively about the EU
## total_experiment$EUextremist <- ifelse(total_experiment$CG24040==1, 1,
##                                        ifelse(total_experiment$CG24040==5, 1,
##                                           ifelse(total_experiment$CG24040==2, 0,
##                                                  ifelse(total_experiment$CG24040==3, 0,
##                                                         ifelse(total_experiment$CG24040==4, 0, NA)))))

## # Not EU knowledgeable: How well do you know the EU's decision-making process? 1=very well to 10=not at all
## total_experiment$unknowledgeable <- ifelse(total_experiment$CG24041<0 , NA,
##                                     ifelse(total_experiment$CG24041>=6 , 1, 0))

## #Disengaged respondents
## total_experiment$disengaged <- ifelse(total_experiment$AA25039>2, 0,
##                                ifelse(total_experiment$AA25039==2, NA, 1))

###################
# Balancing check #
# Fig A1          #
###################
significances <- as.data.frame(matrix(NA, ncol=5, nrow=0))
for (i in 0:8) {
    for (j in 0:8) {
less <- c()
less <- c(less, t.test(x=total_experiment$remain24[total_experiment$expZJ28020==i],
       y=total_experiment$remain24[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$leave24[total_experiment$expZJ28020==i],
       y=total_experiment$leave24[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$DK24[total_experiment$expZJ28020==i],
       y=total_experiment$DK24[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$female[total_experiment$expZJ28020==i],
       y=total_experiment$female[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$age[total_experiment$expZJ28020==i],
       y=total_experiment$age[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$schoolupto10years[total_experiment$expZJ28020==i],
       y=total_experiment$schoolupto10years[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$pro[total_experiment$expZJ28020==i],
       y=total_experiment$pro[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$anti_party[total_experiment$expZJ28020==i],
       y=total_experiment$anti_party[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$east[total_experiment$expZJ28020==i],
       y=total_experiment$east[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
less <- c(less, t.test(x=total_experiment$LR[total_experiment$expZJ28020==i],
       y=total_experiment$LR[total_experiment$expZJ28020==j],
                         alternative="less")$p.value)
greater <- c()
greater <- c(greater, t.test(x=total_experiment$remain24[total_experiment$expZJ28020==i],
       y=total_experiment$remain24[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$leave24[total_experiment$expZJ28020==i],
       y=total_experiment$leave24[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$DK24[total_experiment$expZJ28020==i],
       y=total_experiment$DK24[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$female[total_experiment$expZJ28020==i],
       y=total_experiment$female[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$age[total_experiment$expZJ28020==i],
       y=total_experiment$age[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$schoolupto10years[total_experiment$expZJ28020==i],
       y=total_experiment$schoolupto10years[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$pro[total_experiment$expZJ28020==i],
       y=total_experiment$pro[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$anti_party[total_experiment$expZJ28020==i],
       y=total_experiment$anti_party[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$east[total_experiment$expZJ28020==i],
       y=total_experiment$east[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
greater <- c(greater, t.test(x=total_experiment$LR[total_experiment$expZJ28020==i],
       y=total_experiment$LR[total_experiment$expZJ28020==j],
                         alternative="greater")$p.value)
significances <- rbind(significances, cbind(i, 1:10, less, greater, j))
 }
}
names(significances) <-  c("group", "var", "less", "greater", "target")
significances$group <- factor(significances$group,
                              labels=c("Political\nPro", "Political\nCon",
                                        "Cultural\nPro", "Cultural\nCon",
                                        "Econ\nPro", "Econ\nCon",
                                         "Peace\nPro", "Peace\nCon",
                                  "Control"))
significances$target <- factor(significances$target,
                              labels=c("Political\nPro", "Political\nCon",
                                        "Cultural\nPro", "Cultural\nCon",
                                        "Econ\nPro", "Econ\nCon",
                                         "Peace\nPro", "Peace\nCon",
                                  "Control"))
significances$var <- factor(significances$var,
                            labels=c("Remainers", "Leavers", "Don't\nKnowers'",
                                "Female", "Age", "Schooling", "Pro\nEU",
                                "Eurosceptic\nParty", "East\nGerman", "Left-Right\nPlacement"))
significances$col <- ifelse(significances$less<.05, -1, 0)
significances$col <- ifelse(significances$greater<.05, 1, significances$col)
significances <- significances[significances$group!=significances$target,]

#plot comparing all groups to control only
p <- ggplot(data=significances[significances$target=="Control",], aes(y=var, x=group))
p <- p + theme_bw()
p <- p + theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank())
p <- p + geom_tile(aes(fill=as.factor(col)), colour="black")
p <- p + scale_fill_manual(values=c("gray24", "white", "black"),
                           name="",
                           labels=c("significantly smaller",
                               "not significant", "significantly greater"))
p <- p + ylab("")
p <- p + xlab("")
p <- p + scale_x_discrete(expand=c(0,0))
p <- p + scale_y_discrete(expand=c(0,0))
p
ggsave(file="significance_controls.pdf") #REP: Fig A1


###########################################
# Table Overview of Treatment Group Sizes #
# Tables A1-A9                            #
###########################################
summary.groups <- data.frame(respondents=table(total_experiment$trt.factor))
summary.groups$percent <- summary.groups$respondents.Freq/sum(summary.groups$respondents.Freq)*100
summary.groups$percent.Var1 <- NULL
names(summary.groups) <- c("group", "respondents", "percentage")
summary.groups #REP: Tables A1 and A2


summary.support24 <- data.frame(respondents=table(total_experiment$new.scale24),
                percent=round(table(total_experiment$new.scale24)/nrow(total_experiment)*100,
                    digits=1))
summary.support24$percent.Var1 <- NULL
names(summary.support24) <- c("group", "respondents", "percentage")
summary.support24$variable <- "support24"
summary.support24 #REP: Table A3

summary.support28 <- data.frame(respondents=table(total_experiment$new.scale28),
                percent=round(table(total_experiment$new.scale28)/nrow(total_experiment)*100,
                    digits=1))
summary.support28$percent.Var1 <- NULL
names(summary.support28) <- c("group", "respondents", "percentage")
summary.support28$variable <- "support28"
summary.support28 #REP: Table A4

summary.moderates <- data.frame(respondents=table(total_experiment$EUextremist),
                percent=round(table(total_experiment$EUextremist)/nrow(total_experiment)*100,
                    digits=1))
summary.moderates$percent.Var1 <- NULL
names(summary.moderates) <- c("group", "respondents", "percentage")
summary.moderates$variable <- "moderates"
summary.moderates #REP: Table A5

summary.disengaged <- data.frame(respondents=table(total_experiment$disengaged),
                percent=round(table(total_experiment$disengaged)/nrow(total_experiment)*100,
                    digits=1))
summary.disengaged$percent.Var1 <- NULL
names(summary.disengaged) <- c("group", "respondents", "percentage")
summary.disengaged$variable <- "disengaged"
summary.disengaged #REP: Table A6

summary.unknowledgeable <- data.frame(respondents=table(total_experiment$unknowledgeable),
                percent=round(table(total_experiment$unknowledgeable)/nrow(total_experiment)*100,
                    digits=1))
summary.unknowledgeable$percent.Var1 <- NULL
names(summary.unknowledgeable) <- c("group", "respondents", "percentage")
summary.unknowledgeable$variable <- "unknowledgeable"
summary.unknowledgeable #REP: Table A7

summary.pro.dum <- data.frame(respondents=table(total_experiment$pro.dum),
                percent=round(table(total_experiment$pro.dum)/nrow(total_experiment)*100,
                    digits=1))
summary.pro.dum$percent.Var1 <- NULL
names(summary.pro.dum) <- c("group", "respondents", "percentage")
summary.pro.dum$variable <- "pro.dum"
summary.pro.dum #REP: Table A8

summary.anti_party <- data.frame(respondents=table(total_experiment$anti_party),
                percent=round(table(total_experiment$anti_party)/nrow(total_experiment)*100,
                    digits=1))
summary.anti_party$percent.Var1 <- NULL
names(summary.anti_party) <- c("group", "respondents", "percentage")
summary.anti_party$variable <- "anti_party"
summary.anti_party #REP: Table A9

#########################################################
# regression plots based on interactions and subsamples #
# Fig 4 and Tab A13-A17                                 #
#########################################################

total_experiment$tr_control   = ifelse(total_experiment$expZJ28020==0, 1,0)
total_experiment$tr_pol_pos   = ifelse(total_experiment$expZJ28020==1, 1,0)
total_experiment$tr_pol_neg   = ifelse(total_experiment$expZJ28020==2, 1,0)
total_experiment$tr_cul_pos   = ifelse(total_experiment$expZJ28020==3, 1,0)
total_experiment$tr_cul_neg   = ifelse(total_experiment$expZJ28020==4, 1,0)
total_experiment$tr_econ_pos  = ifelse(total_experiment$expZJ28020==5, 1,0)
total_experiment$tr_econ_neg  = ifelse(total_experiment$expZJ28020==6, 1,0)
total_experiment$tr_peace_pos = ifelse(total_experiment$expZJ28020==7, 1,0)
total_experiment$tr_peace_neg = ifelse(total_experiment$expZJ28020==8, 1,0)
total_experiment <- as.data.table(total_experiment)

treatments=c("tr_pol_pos","tr_pol_neg" , "tr_cul_pos", "tr_cul_neg",
    "tr_econ_pos", "tr_econ_neg", "tr_peace_pos", "tr_peace_neg")
trnames=c("Political, positive","Political, negative" , "Cultural, positive",
    "Cultural, negative", "Economic, positive", "Economic, negative",
    "Peace, positive", "Peace, negative")
trnames.short <-c("+Political","-Political" , "+Cultural",
                  "-Cultural", "+Economic", "-Economic", "+Peace", "-Peace")

set.seed(12736)
all.models <- list()
all.data <- list()


#EUextremist
models <- list()
single.coef <- vector(mode="double",length=length(treatments))
single.lower <- vector(mode="double",length=length(treatments))
single.upper <- vector(mode="double",length=length(treatments))
inter.coef <- vector(mode="double",length=length(treatments))
inter.lower <- vector(mode="double",length=length(treatments))
inter.upper <- vector(mode="double",length=length(treatments))
joint.coef <- vector(mode="double",length=length(treatments))
joint.lower <- vector(mode="double",length=length(treatments))
joint.upper <- vector(mode="double",length=length(treatments))
for (i in 1:8) {
    curr_treatment_var=paste(treatments[i], sep="")
    models[[i]] <- lm(Dif_remain ~ get(curr_treatment_var[1])*EUextremist,
                      data=total_experiment[expZJ28020==i|expZJ28020==0])
    betas <- mvrnorm(1000, coef(models[[i]]), vcov(models[[i]]))
    single.coef[i] <- mean(betas[,2])
    single.lower[i] <- quantile(betas[,2], probs=.025)
    single.upper[i] <- quantile(betas[,2], probs=.975)
    inter.coef[i] <- mean(betas[,4])
    inter.lower[i] <- quantile(betas[,4], probs=.025)
    inter.upper[i] <- quantile(betas[,4], probs=.975)
    joint.coef[i] <- mean(betas[,4]+betas[,2])
    joint.lower[i] <- quantile(betas[,4]+betas[,2], probs=.025)
    joint.upper[i] <- quantile(betas[,4]+betas[,2], probs=.975)
}
all.models[[1]] <- models
all.data[[1]] <- data.table(single.coef=single.coef, single.lower=single.lower, single.upper=single.upper,
                            inter.coef=inter.coef, inter.lower=inter.lower, inter.upper=inter.upper,
                            joint.coef=joint.coef, joint.lower=joint.lower, joint.upper=joint.upper,
                            treatment=1:8,
                            variable="EUextremist"
                            )
stargazer(models[[1]], models[[2]], models[[3]], models[[4]],
          models[[5]], models[[6]], models[[7]], models[[8]],
          align=TRUE, type = "latex")
#REP: Table A13

#disengaged
models <- list()
single.coef <- vector(mode="double",length=length(treatments))
single.lower <- vector(mode="double",length=length(treatments))
single.upper <- vector(mode="double",length=length(treatments))
inter.coef <- vector(mode="double",length=length(treatments))
inter.lower <- vector(mode="double",length=length(treatments))
inter.upper <- vector(mode="double",length=length(treatments))
joint.coef <- vector(mode="double",length=length(treatments))
joint.lower <- vector(mode="double",length=length(treatments))
joint.upper <- vector(mode="double",length=length(treatments))
for (i in 1:8) {
    curr_treatment_var=paste(treatments[i], sep="")
    models[[i]] <- lm(Dif_remain ~ get(curr_treatment_var[1])*disengaged,
                      data=total_experiment[expZJ28020==i|expZJ28020==0])
    betas <- mvrnorm(1000, coef(models[[i]]), vcov(models[[i]]))
    single.coef[i] <- mean(betas[,2])
    single.lower[i] <- quantile(betas[,2], probs=.025)
    single.upper[i] <- quantile(betas[,2], probs=.975)
    inter.coef[i] <- mean(betas[,4])
    inter.lower[i] <- quantile(betas[,4], probs=.025)
    inter.upper[i] <- quantile(betas[,4], probs=.975)
    joint.coef[i] <- mean(betas[,4]+betas[,2])
    joint.lower[i] <- quantile(betas[,4]+betas[,2], probs=.025)
    joint.upper[i] <- quantile(betas[,4]+betas[,2], probs=.975)
}
all.models[[2]] <- models
all.data[[2]] <- data.table(single.coef=single.coef, single.lower=single.lower, single.upper=single.upper,
                            inter.coef=inter.coef, inter.lower=inter.lower, inter.upper=inter.upper,
                            joint.coef=joint.coef, joint.lower=joint.lower, joint.upper=joint.upper,
                            treatment=1:8,
                            variable="disengaged"
                            )
stargazer(models[[1]], models[[2]], models[[3]], models[[4]],
          models[[5]], models[[6]], models[[7]], models[[8]],
          align=TRUE, type = "latex")
#REP: Table A14

#unknowledgeable
models <- list()
single.coef <- vector(mode="double",length=length(treatments))
single.lower <- vector(mode="double",length=length(treatments))
single.upper <- vector(mode="double",length=length(treatments))
inter.coef <- vector(mode="double",length=length(treatments))
inter.lower <- vector(mode="double",length=length(treatments))
inter.upper <- vector(mode="double",length=length(treatments))
joint.coef <- vector(mode="double",length=length(treatments))
joint.lower <- vector(mode="double",length=length(treatments))
joint.upper <- vector(mode="double",length=length(treatments))
for (i in 1:8) {
    curr_treatment_var=paste(treatments[i], sep="")
    models[[i]] <- lm(Dif_remain ~ get(curr_treatment_var[1])*unknowledgeable,
                      data=total_experiment[expZJ28020==i|expZJ28020==0])
     betas <- mvrnorm(1000, coef(models[[i]]), vcov(models[[i]]))
    single.coef[i] <- mean(betas[,2])
    single.lower[i] <- quantile(betas[,2], probs=.025)
    single.upper[i] <- quantile(betas[,2], probs=.975)
    inter.coef[i] <- mean(betas[,4])
    inter.lower[i] <- quantile(betas[,4], probs=.025)
    inter.upper[i] <- quantile(betas[,4], probs=.975)
    joint.coef[i] <- mean(betas[,4]+betas[,2])
    joint.lower[i] <- quantile(betas[,4]+betas[,2], probs=.025)
    joint.upper[i] <- quantile(betas[,4]+betas[,2], probs=.975)
}
all.models[[3]] <- models
all.data[[3]] <- data.table(single.coef=single.coef, single.lower=single.lower, single.upper=single.upper,
                            inter.coef=inter.coef, inter.lower=inter.lower, inter.upper=inter.upper,
                            joint.coef=joint.coef, joint.lower=joint.lower, joint.upper=joint.upper,
                            treatment=1:8,
                            variable="unknowledgeable"
                            )
stargazer(models[[1]], models[[2]], models[[3]], models[[4]],
          models[[5]], models[[6]], models[[7]], models[[8]],
          align=TRUE, type = "latex")
#REP: Table A15

#pro.dum
models <- list()
single.coef <- vector(mode="double",length=length(treatments))
single.lower <- vector(mode="double",length=length(treatments))
single.upper <- vector(mode="double",length=length(treatments))
inter.coef <- vector(mode="double",length=length(treatments))
inter.lower <- vector(mode="double",length=length(treatments))
inter.upper <- vector(mode="double",length=length(treatments))
joint.coef <- vector(mode="double",length=length(treatments))
joint.lower <- vector(mode="double",length=length(treatments))
joint.upper <- vector(mode="double",length=length(treatments))
for (i in 1:8) {
    curr_treatment_var=paste(treatments[i], sep="")
    models[[i]] <- lm(Dif_remain ~ get(curr_treatment_var[1])*pro.dum,
                      data=total_experiment[expZJ28020==i|expZJ28020==0])
     betas <- mvrnorm(1000, coef(models[[i]]), vcov(models[[i]]))
    single.coef[i] <- mean(betas[,2])
    single.lower[i] <- quantile(betas[,2], probs=.025)
    single.upper[i] <- quantile(betas[,2], probs=.975)
    inter.coef[i] <- mean(betas[,4])
    inter.lower[i] <- quantile(betas[,4], probs=.025)
    inter.upper[i] <- quantile(betas[,4], probs=.975)
    joint.coef[i] <- mean(betas[,4]+betas[,2])
    joint.lower[i] <- quantile(betas[,4]+betas[,2], probs=.025)
    joint.upper[i] <- quantile(betas[,4]+betas[,2], probs=.975)
}
all.models[[4]] <- models
all.data[[4]] <- data.table(single.coef=single.coef, single.lower=single.lower, single.upper=single.upper,
                            inter.coef=inter.coef, inter.lower=inter.lower, inter.upper=inter.upper,
                            joint.coef=joint.coef, joint.lower=joint.lower, joint.upper=joint.upper,
                            treatment=1:8,
                            variable="pro.dum"
                            )
stargazer(models[[1]], models[[2]], models[[3]], models[[4]],
          models[[5]], models[[6]], models[[7]], models[[8]],
          align=TRUE, type = "latex")
#REP: Table A16

#anti_party
models <- list()
single.coef <- vector(mode="double",length=length(treatments))
single.lower <- vector(mode="double",length=length(treatments))
single.upper <- vector(mode="double",length=length(treatments))
inter.coef <- vector(mode="double",length=length(treatments))
inter.lower <- vector(mode="double",length=length(treatments))
inter.upper <- vector(mode="double",length=length(treatments))
joint.coef <- vector(mode="double",length=length(treatments))
joint.lower <- vector(mode="double",length=length(treatments))
joint.upper <- vector(mode="double",length=length(treatments))
for (i in 1:8) {
    curr_treatment_var=paste(treatments[i], sep="")
    models[[i]] <- lm(Dif_remain ~ get(curr_treatment_var[1])*anti_party,
                      data=total_experiment[expZJ28020==i|expZJ28020==0])
    betas <- mvrnorm(1000, coef(models[[i]]), vcov(models[[i]]))
    single.coef[i] <- mean(betas[,2])
    single.lower[i] <- quantile(betas[,2], probs=.025)
    single.upper[i] <- quantile(betas[,2], probs=.975)
    inter.coef[i] <- mean(betas[,4])
    inter.lower[i] <- quantile(betas[,4], probs=.025)
    inter.upper[i] <- quantile(betas[,4], probs=.975)
    joint.coef[i] <- mean(betas[,4]+betas[,2])
    joint.lower[i] <- quantile(betas[,4]+betas[,2], probs=.025)
    joint.upper[i] <- quantile(betas[,4]+betas[,2], probs=.975)
}
all.models[[5]] <- models
all.data[[5]] <- data.table(single.coef=single.coef, single.lower=single.lower, single.upper=single.upper,
                            inter.coef=inter.coef, inter.lower=inter.lower, inter.upper=inter.upper,
                            joint.coef=joint.coef, joint.lower=joint.lower, joint.upper=joint.upper,
                            treatment=1:8,
                            variable="anti_party"
                            )
stargazer(models[[1]], models[[2]], models[[3]], models[[4]],
          models[[5]], models[[6]], models[[7]], models[[8]],
          align=TRUE, type = "latex")
#REP: Table A17

pdata <- rbindlist(all.data)
pdata$y <- ifelse(pdata$variable=="EUextremist",14,
           ifelse(pdata$variable=="disengaged",11,
           ifelse(pdata$variable=="unknowledgeable",8,
           ifelse(pdata$variable=="pro.dum",5,
           ifelse(pdata$variable=="anti_party",2,NA)))))
pdata$tr.fct <- factor(pdata$treatment,
                       labels=c("+Political",
                                "-Political",
                                "+Cultural",
                                "-Cultural",
                                "+Economic",
                                "-Economic",
                                "+Peace",
                                "-Peace"))
pdata$y <- ifelse(pdata$variable=="EUextremist",19,
           ifelse(pdata$variable=="disengaged",15,
           ifelse(pdata$variable=="unknowledgeable",11,
           ifelse(pdata$variable=="pro.dum",7,
           ifelse(pdata$variable=="anti_party",3,NA)))))
p <- ggplot(pdata, aes(x=y))
p <- p + coord_flip()
p <- p + facet_grid(~tr.fct)
p <- p + scale_x_continuous(breaks=c(1:3, 5:7, 9:11, 13:15, 17:19),
                            labels=c("   Difference",
                                "Not Eurosceptic Party Supporter",
                                "Eurosceptic Party Supporter",
                                "   Difference",
                                     "No Pro EU Attitude",
                                "Pro EU Attitude",
                                "   Difference",
                                     "EU knowledgeable",
                                "Not EU knowledgeable",
                                "   Difference",
                                     "Engaged",
                                "Disengaged",
                                "   Difference",
                                     "EU moderate",
                                "EU extremist"
                                     ))
p <- p + geom_hline(yintercept=0, color="grey")
p <- p + geom_vline(xintercept=c(4, 8, 12, 16), color="grey")
p <- p + geom_linerange(aes(ymax = single.upper, ymin = single.lower, x=y-1),
                        position=position_dodge(width=.1), size=1)
p <- p + geom_point(aes(y=single.coef, x=y-1), position=position_dodge(width=.1), size=3, shape=16)
p <- p + geom_linerange(aes(ymax = joint.upper, ymin = joint.lower, x=y),
                        position=position_dodge(width=.1), size=1)
p <- p + geom_point(aes(y=joint.coef, x=y), position=position_dodge(width=.1), size=3, shape=16)
p <- p + geom_linerange(aes(ymax = inter.upper, ymin = inter.lower, x=y-2),
                        position=position_dodge(width=.1), size=1, color="grey")
p <- p + geom_point(aes(y=inter.coef, x=y-2), position=position_dodge(width=.1), size=3, shape=16, , color="grey")
p <- p + xlab("")
p <- p + ylab("")
p <- p + theme_bw()
p <- p + theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
               legend.position="bottom", strip.background = element_blank(),
               axis.text.x = element_text(angle = 45, hjust = 1),
               axis.ticks.y=element_blank())
p <- p + theme(text = element_text(size=25)) #font size
p
ggsave("Hyps_interaction.png", units="mm", width=508, height=263.5)
#REP: Fig 4

###################################################
# Regression with levels instead of changes as DV #
# Tab A19                                         #
###################################################
long <- melt(total_experiment[is.na(new.scale24)==FALSE&is.na(new.scale28)==FALSE],
measure.vars=c("new.scale24", "new.scale28"),
id.vars=c("expZJ28020", "female", "east", "schoolupto10years", "age", "pro", "anti_party", "LR"))
long$w28 <- ifelse(long$variable=="new.scale28", 1, 0)

linear.7<-lm(value~as.factor(expZJ28020)*w28, data=long)

linear.8<-lm(value~as.factor(expZJ28020)*w28
+female
+east
+schoolupto10years
+as.factor(age)
+pro
+anti_party
+LR,
data=long)

olog.5<-polr(as.factor(value)~as.factor(expZJ28020)*w28,
data=long)

olog.6<-polr(as.factor(value)~as.factor(expZJ28020)*w28
+female
+east
+schoolupto10years
+as.factor(age)
+pro
+anti_party
+LR,
data=long)

stargazer(linear.7, linear.8, olog.5, olog.6, title="Simple treatment effects", align=TRUE, type = "latex")
#REP: Table A19

##############################
# Models with all treatments #
# Fig 2a, Fig 2b and Tab A12 #
##############################
#linear models
lmR1<-lm(Dif_remain~as.factor(expZJ28020),
             data=total_experiment)
lmL1<-lm(Dif_leave~ as.factor(expZJ28020),
             data=total_experiment)
lmDK1<-lm(Dif_dk~ as.factor(expZJ28020),
             data=total_experiment)
lmR2<-lm(Dif_remain~ as.factor(expZJ28020)
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
lmL2<-lm(Dif_leave~ as.factor(expZJ28020)
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
lmDK2<-lm(Dif_dk~ as.factor(expZJ28020)
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
stargazer(lmR1, lmL1, lmDK1, lmR2, lmL2, lmDK2,
          title="OLS results", align=TRUE, type = "latex") #REP: Table A12
#REP: Table A12

#plots based on lmR1
#plot with coefficients based on lmR1gt
pdata <- data.table(x=coef(lmR1)[c(3,9,5,7)],
                    lower=coef(lmR1)[c(3,9,5,7)]-(1.96*sqrt(diag(vcov(lmR1)))[c(3,9,5,7)]),
                    upper=coef(lmR1)[c(3,9,5,7)]+(1.96*sqrt(diag(vcov(lmR1)))[c(3,9,5,7)]),
                    label=c("-Political", "-Peace", "-Cultural", "-Economic"),
                    order=c(4, 3, 2, 1))
p <- ggplot(pdata, aes(x=order, y=x))
p <- p + scale_x_continuous(breaks=4:1, labels=pdata$label)
p <- p + coord_flip()
p <- p + geom_hline(yintercept=0, color="grey")
p <- p + geom_linerange(aes(ymax = upper, ymin = lower),
                       position=position_dodge(width=.1), size=1)
p <- p + geom_point(position=position_dodge(width=.1), size=3, shape=16)
p <- p + xlab("")
p <- p + ylab("")
p <- p + theme_bw()
p <- p + theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
               legend.position="bottom", strip.background = element_blank(),
               text = element_text(size=13), axis.text.x = element_text(angle = 45, hjust = 1),
               axis.ticks.y=element_blank())
p
ggsave("Single_Treatments1.pdf", width=259.6,  units="mm")
#REP: Fig 2a

##########################
# ODDS-RATIOS            #
# -Pol vs Control Group  #
##########################
table(total_experiment$remain28, total_experiment$trt.factor=="Political Con")

round((48/205)/(304/1657),2) #odds ratio between being not remain when getting -Pol/-Peace rather than control


#plot with F-test p-values based on lmR1gt
pdata <- as.data.table(rbind(
    cbind(1, 1, 1),
    cbind(1, 2,
          linearHypothesis(lmR1, "as.factor(expZJ28020)2=as.factor(expZJ28020)8")$"Pr(>F)"[2]),
    cbind(1, 3,
          linearHypothesis(lmR1, "as.factor(expZJ28020)2=as.factor(expZJ28020)6")$"Pr(>F)"[2]),
    cbind(1, 4,
          linearHypothesis(lmR1, "as.factor(expZJ28020)2=as.factor(expZJ28020)4")$"Pr(>F)"[2]),
    cbind(2, 1, NA),
    cbind(2, 2, 1),
    cbind(2, 3,
          linearHypothesis(lmR1, "as.factor(expZJ28020)8=as.factor(expZJ28020)6")$"Pr(>F)"[2]),
    cbind(2, 4,
          linearHypothesis(lmR1, "as.factor(expZJ28020)8=as.factor(expZJ28020)4")$"Pr(>F)"[2]),
    cbind(4, 1, NA),
    cbind(4, 2, NA),
    cbind(4, 3, NA),
    cbind(4, 4, 1),
    cbind(3, 1, NA),
    cbind(3, 2, NA),
    cbind(3, 4,
          linearHypothesis(lmR1, "as.factor(expZJ28020)4=as.factor(expZJ28020)6")$"Pr(>F)"[2]),
    cbind(3, 3, 1)))
pdata$col <- ifelse(pdata$V3<.05, 1, 0)
pdata$col <- ifelse(is.na(pdata$V3<.05), 0, pdata$col)
p <- ggplot(data=pdata, aes(y=V1, x=V2))
p <- p + theme_bw()
p <- p + theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
               axis.ticks.y=element_blank(), axis.ticks.x=element_blank())
p <- p + geom_tile(aes(fill=as.factor(col)), colour="black", data=pdata[is.na(V3)==FALSE])
p <- p + scale_y_continuous(breaks=1:4, labels=c("-Political", "-Peace", "-Economic", "-Cultural"))
p <- p + scale_x_continuous(breaks=1:4, labels=c("-Political", "-Peace", "-Economic", "-Cultural"))
p <- p + xlab("")
p <- p + ylab("")
p <- p + scale_fill_manual(values=c("white", "grey"),
                           name="",
                           labels=c("no difference", "significantly different"))
p
ggsave("Single_Treatments2.pdf", units="mm")
#REP: Fig 2b

#corresponding F-test
          linearHypothesis(lmR1, "as.factor(expZJ28020)2=as.factor(expZJ28020)8") #pol/peace
          linearHypothesis(lmR1, "as.factor(expZJ28020)2=as.factor(expZJ28020)6") #pol/econ
          linearHypothesis(lmR1, "as.factor(expZJ28020)2=as.factor(expZJ28020)4") #pol/cult
          linearHypothesis(lmR1, "as.factor(expZJ28020)8=as.factor(expZJ28020)6") #peace/econ
          linearHypothesis(lmR1, "as.factor(expZJ28020)8=as.factor(expZJ28020)4") #peace/cult
          linearHypothesis(lmR1, "as.factor(expZJ28020)4=as.factor(expZJ28020)6") #cult/econ


##############################
# ODDS-RATIOS                #
# -Pol/-Peace vs -Cul/-Econ  #
##############################







#################
# Ologit Models #
# Tab A18       #
#################
ologR1<-polr(as.factor(Dif_remain)~as.factor(expZJ28020),
             data=total_experiment)
ologL1<-polr(as.factor(Dif_leave)~ as.factor(expZJ28020),
             data=total_experiment)
ologDK1<-polr(as.factor(Dif_dk)~ as.factor(expZJ28020),
             data=total_experiment)
ologR2<-polr(as.factor(Dif_remain)~ as.factor(expZJ28020)
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
ologL2<-polr(as.factor(Dif_leave)~ as.factor(expZJ28020)
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
ologDK2<-polr(as.factor(Dif_dk)~ as.factor(expZJ28020)
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
stargazer(ologR1, ologL1, ologDK1, ologR2, ologL2, ologDK2,
          title="Simple treatment effects", align=TRUE, type = "latex") #REP: Table A18

#########################################
# Positive and Negative Treatments only #
# Fig 1a and Tab A10                    #
#########################################
total_experiment$pos_treat <- ifelse(total_experiment$expZJ28020 %in% c(1,3,5,7), 1, 0)
total_experiment$neg_treat <- ifelse(total_experiment$expZJ28020 %in% c(2,4,6,8), 1, 0)


lmR1pn<-lm(Dif_remain~pos_treat+neg_treat,
           data=total_experiment)
lmL1pn<-lm(Dif_leave~ pos_treat+neg_treat,
             data=total_experiment)
lmDK1pn<-lm(Dif_dk~ pos_treat+neg_treat,
             data=total_experiment)
lmR2pn<-lm(Dif_remain~ pos_treat+neg_treat
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
lmL2pn<-lm(Dif_leave~ pos_treat+neg_treat
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
lmDK2pn<-lm(Dif_dk~ pos_treat+neg_treat
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
stargazer(lmR1pn, lmL1pn, lmDK1pn, lmR2pn, lmL2pn, lmDK2pn,
          title="OLS results", align=TRUE, type = "latex") #REP: Table A14
#REP: Table A10

#plot with coefficients and their difference based on lmR1pn
dif <- difference(lmR1pn, "pos_treat", "neg_treat")
linearHypothesis(lmR1pn, "pos_treat=neg_treat") #corresponding F-test
pdata <- data.table(x=c(coef(lmR1pn)[2:3], dif$mean),
                    lower=c(coef(lmR1pn)[2:3]-(1.96*sqrt(diag(vcov(lmR1pn)))[2:3]), dif$lower),
                    upper=c(coef(lmR1pn)[2:3]+(1.96*sqrt(diag(vcov(lmR1pn)))[2:3]), dif$upper),
                    label=as.factor(c("Positive Treatments", "Negative Treatments", "Difference")),
                    color=c(1, 1, 2))
p <- ggplot(pdata, aes(x=label, y=x))
p <- p + coord_flip()
p <- p + scale_color_manual(values=c("black", "grey"), "", guide=FALSE)
p <- p + geom_hline(yintercept=0, color="grey")
p <- p + geom_linerange(aes(ymax = upper, ymin = lower, color=as.factor(color)),
                       position=position_dodge(width=.1), size=1)
p <- p + geom_point(position=position_dodge(width=.1), aes(color=as.factor(color)),
                                                           size=3, shape=16)
p <- p + xlab("")
p <- p + ylab("")
p <- p + theme_bw()
p <- p + theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
               legend.position="bottom", strip.background = element_blank(),
               axis.text.x = element_text(angle = 45, hjust = 1),
               axis.ticks.y=element_blank())
p <- p + theme(text = element_text(size=20)) #font size
p
ggsave("Pos_Neg_Comparison.pdf", width=259.6,  units="mm") #REP: Fig 1a


######################
# Grouped Treatments #
# Fig 1b and Tab A11 #
######################
total_experiment$peace.political.pro <- ifelse(total_experiment$expZJ28020 %in% c(1,7), 1, 0)
total_experiment$peace.political.con <- ifelse(total_experiment$expZJ28020 %in% c(2,8), 1, 0)
total_experiment$econ.cultural.pro <- ifelse(total_experiment$expZJ28020 %in% c(3,5), 1, 0)
total_experiment$econ.cultural.con <- ifelse(total_experiment$expZJ28020 %in% c(4,6), 1, 0)

lmR1gt <- lm(Dif_remain~peace.political.pro+peace.political.con+econ.cultural.pro+econ.cultural.con,
             data=total_experiment)
lmL1gt <- lm(Dif_leave~ peace.political.pro+peace.political.con+econ.cultural.pro+econ.cultural.con,
             data=total_experiment)
lmDK1gt <- lm(Dif_dk~ peace.political.pro+peace.political.con+econ.cultural.pro+econ.cultural.con,
             data=total_experiment)
lmR2gt <- lm(Dif_remain~ peace.political.pro+peace.political.con+econ.cultural.pro+econ.cultural.con
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
lmL2gt <- lm(Dif_leave~ peace.political.pro+peace.political.con+econ.cultural.pro+econ.cultural.con
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
lmDK2gt <- lm(Dif_dk~ peace.political.pro+peace.political.con+econ.cultural.pro+econ.cultural.con
             +female
             +east
             +schoolupto10years
             +as.factor(age)
             +pro
             +anti_party
             +LR,
             data=total_experiment)
stargazer(lmR1gt, lmL1gt, lmDK1gt, lmR2gt, lmL2gt, lmDK2gt,
          title="OLS results", align=TRUE, type = "latex")
# REP: Tab A11

#plot with coefficients and their difference based on lmR1gt
dif <- difference(lmR1gt, "peace.political.con", "econ.cultural.con")
linearHypothesis(lmR1gt, "peace.political.con=econ.cultural.con") #corresponding F-test
pdata <- data.table(x=c(coef(lmR1gt)[c(3,5)], dif$mean),
                    lower=c(coef(lmR1gt)[c(3,5)]-(1.96*sqrt(diag(vcov(lmR1gt)))[c(3,5)]),
                            dif$lower),
                    upper=c(coef(lmR1gt)[c(3,5)]+(1.96*sqrt(diag(vcov(lmR1gt)))[c(3,5)]),
                            dif$upper),
                    label=c("-Political/-Peace", "-Cultural/-Economic", "Difference"),
                    order=c(3,2,1),
                    color=c(1, 1, 2))
p <- ggplot(pdata, aes(x=order, y=x))
p <- p + scale_x_continuous(breaks=3:1, labels=pdata$label)
p <- p + expand_limits(y=c(-.105, .105))
p <- p + coord_flip()
p <- p + scale_color_manual(values=c("black", "grey"), "", guide=FALSE)
p <- p + geom_hline(yintercept=0, color="grey")
p <- p + geom_linerange(aes(ymax = upper, ymin = lower, color=as.factor(color)),
                       position=position_dodge(width=.1), size=1)
p <- p + geom_point(position=position_dodge(width=.1), aes(color=as.factor(color)),
                                                           size=3, shape=16)
p <- p + xlab("")
p <- p + ylab("")
p <- p + theme_bw()
p <- p + theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
               legend.position="bottom", strip.background = element_blank(),
               axis.text.x = element_text(angle = 45, hjust = 1),
               axis.ticks.y=element_blank())
p <- p + theme(text = element_text(size=20)) #font size
p
ggsave("Grouped_Comparison.pdf", width=260,  units="mm")
#REP: Fig 1b

##############################
# ODDS-RATIOS                #
# -Pol/-Peace vs -Cul/-Econ  #
##############################
table(total_experiment$remain28, total_experiment$peace.political.con)
table(total_experiment$remain28, total_experiment$econ.cultural.con)

round((91/415)/(84/412),2) #odds ratio between being not remain when getting -Pol/-Peace rather than -Eco/-Cul

##############
# Trend plot #
# Fig 3      #
##############
pdata <- as.data.table(total_experiment)
pdata <- pdata[,list(remain24, remain28, leave24, leave28, DK24, DK28, expZJ28020)]
pdata <- pdata[,c("Mremain24", "Mremain28", "Mleave24", "Mleave28", "MDK24", "MDK28"):=list(
                    mean(remain24),
                    mean(remain28),
                    mean(leave24),
                    mean(leave28),
                    mean(DK24),
                    mean(DK28)
                ), expZJ28020]
pdata <- unique(pdata[,list(Mremain24, Mremain28, Mleave24, Mleave28, MDK24, MDK28, expZJ28020)])
pdata <- data.table(first=c(pdata$Mremain24, pdata$Mleave24,  pdata$MDK24),
                    second=c(pdata$Mremain28, pdata$Mleave28,  pdata$MDK28),
                    expZJ28020=rep(pdata$expZJ28020, 3),
                    type=rep(c(1, 2, 3), each=9))
pdata$type <- factor(pdata$type, labels=c("Remainers", "Leavers", "Don't Know"))
pdata$label <- factor(pdata$expZJ28020,
                      labels=c("Control",
                               "+Political", "-Political",
                               "+Cultural", "-Cultural",
                               "+Economic", "-Economic",
                               "+Peace", "-Peace"))
pdata$labelx <- pdata$second
pdata$labelx <- ifelse(pdata$type=="Remainers"&pdata$label=="-Economic", pdata$labelx+.002,
                      pdata$labelx)
pdata$labelx <- ifelse(pdata$type=="Remainers"&pdata$label=="+Economic", pdata$labelx+.002,
                      pdata$labelx)
pdata$labelx <- ifelse(pdata$type=="Leavers"&pdata$label=="-Political", pdata$labelx+.001,
                      pdata$labelx)
pdata$labelx <- ifelse(pdata$type=="Don't Know"&pdata$label=="-Political", pdata$labelx-.0005,
                      pdata$labelx)
pdata$labelx <- ifelse(pdata$type=="Don't Know"&pdata$label=="+Economic", pdata$labelx+.0002,
                      pdata$labelx)
pdata$labelx <- ifelse(pdata$type=="Don't Know"&pdata$label=="-Economic", pdata$labelx+.002,
                      pdata$labelx)
pdata$labelx <- ifelse(pdata$type=="Don't Know"&pdata$label=="-Peace", pdata$labelx-.0005,
                      pdata$labelx)
pdata$labelx <- ifelse(pdata$type=="Don't Know"&pdata$label=="+Peace", pdata$labelx-.001,
                      pdata$labelx)

p <- ggplot(pdata)
p <- p + facet_wrap(~type, scales="free_y")
p <- p + geom_segment(aes(x=1, xend=2, y=first*100, yend=second*100))
p <- p + geom_segment(aes(x=1, xend=2, y=first*100, yend=second*100), size=1.2,
                     data=pdata[label=="-Political"])
p <- p + geom_text(aes(x=2.02, y=labelx*100, label=label), hjust = 0,
                  data=pdata[label!="-Political"], size=5) #font size labels in plot
p <- p + geom_text(aes(x=2.02, y=labelx*100, label=label), fontface="bold", hjust = 0,
                  data=pdata[label=="-Political"], size=5) #font size labels in plot
p <- p + theme_bw()
p <- p + theme(legend.title=element_blank())
p <- p + theme(text = element_text(size=20)) #font size
p <- p + ylab("Share of Respondents in %")
p <- p + scale_x_discrete("", breaks=c(1,2), labels=c("July\n2016","March\n2017"), limits=c(1,2),
                         expand=expand_scale(add=c(.05,.4)))
p
ggsave("trend_pol_con_highlight.pdf", height=263.5, width=508,  units="mm")
#REP: Fig 3


################################################################
## sessionInfo() on the machine this R-script was tested with: #
################################################################
## R version 3.6.2 (2019-12-12)
## Platform: x86_64-w64-mingw32/x64 (64-bit)
## Running under: Windows 10 x64 (build 17763)

## Matrix products: default

## locale:
## [1] LC_COLLATE=German_Germany.1252  LC_CTYPE=German_Germany.1252
## [3] LC_MONETARY=German_Germany.1252 LC_NUMERIC=C
## [5] LC_TIME=German_Germany.1252

## attached base packages:
## [1] stats     graphics  grDevices utils     datasets  methods   base

## other attached packages:
## [1] car_3.0-6         carData_3.0-3     data.table_1.12.8 stargazer_5.2.2
## [5] MASS_7.3-51.5     readstata13_0.9.2 ggplot2_3.2.1     reshape2_1.4.3
## [9] dplyr_0.8.3

## loaded via a namespace (and not attached):
##  [1] zip_2.0.4        Rcpp_1.0.3       cellranger_1.1.0 pillar_1.4.3
##  [5] compiler_3.6.2   plyr_1.8.5       forcats_0.4.0    tools_3.6.2
##  [9] digest_0.6.23    zeallot_0.1.0    lifecycle_0.1.0  tibble_2.1.3
## [13] gtable_0.3.0     pkgconfig_2.0.3  rlang_0.4.2      openxlsx_4.1.4
## [17] curl_4.3         haven_2.2.0      rio_0.5.16       withr_2.1.2
## [21] stringr_1.4.0    vctrs_0.2.1      hms_0.5.3        grid_3.6.2
## [25] tidyselect_0.2.5 glue_1.3.1       R6_2.4.1         readxl_1.3.1
## [29] foreign_0.8-74   farver_2.0.1     purrr_0.3.3      magrittr_1.5
## [33] backports_1.1.5  scales_1.1.0     abind_1.4-5      assertthat_0.2.1
## [37] colorspace_1.4-1 labeling_0.3     stringi_1.4.4    lazyeval_0.2.2
## [41] munsell_0.5.0    crayon_1.3.4
